#!/bin/tclsh

set tmpdir /tmp/lcd_msg
if {![file isdirectory $tmpdir]} then {
    file mkdir $tmpdir
}
set dateformat "%d%m%y"
set timeformat "%H%M"


proc getTimeout { param } {
  set value [string range $param 0 end-1]
  set unit [string index $param end]
  switch "[string tolower $unit]" {
    "s" { expr { $value + [clock seconds] } }
    "m" { expr { $value*60 + [clock seconds] } }
    "h" { expr { $value*3600 + [clock seconds] } }
    "d" { expr { $value*86400 + [clock seconds] } }
    "t" { if { [catch {clock scan $value} sTime]} then {return 0}
          if { $sTime <= [clock seconds] } then {set sTime [clock scan "1 day" -base $sTime]}
          return $sTime
        }
  }
}

proc parseParams { params } {
  upvar msgData msgData
  foreach param $params {
    set paramdata [string range $param 1 end]
    regsub -nocase -all {[^a-z0-9]+} $paramdata "-" descriptor
    switch [string index $param 0] {
      "m" { set msgData(to_message) [getTimeout $paramdata] }
      "b" { set msgData(to_backlight) [getTimeout $paramdata] }
      "g" { set msgData(group) [string range $descriptor 0 5] }
      "i" { set msgData(message_id) [string range $descriptor 0 19] }
      "a" { set msgData(to_alarm) [getTimeout $paramdata] }
      "s" { set msgData(to_service) [getTimeout $paramdata] }
      "S" { set msgData(to_sblink) [getTimeout $paramdata] }
      "f" { set msgData(flash) $paramdata }
      "d" { set msgData(on_delete) $paramdata }
      "r" { set msgData(rolling) $paramdata }
    }
  }
}

#get message data
set msglines [eval join {$argv} " "]
#process the message data

#init parameters
set msgData(to_message) 0
set msgData(to_backlight) 0
set msgData(to_alarm) 0
set msgData(to_service) 0
set msgData(to_sblink) 0
set msgData(flash) 0
set msgData(group) msg
set msgData(message_id) 0
set msgData(rolling) 0
set msgData(on_delete) "exec $tmpdir/on_delete.sh"

#read parameters
if {[regexp {^[$]([^|]+)[$]} $msglines dummy params]} then {
  parseParams $params
  set msglines [string range $msglines [expr {[string length $params]+2}] end]
}

#delete previous copies of file
set msgid $msgData(message_id)
set groupname $msgData(group)
set overwrite 1
source /www/addons/lcd_msg/delete_msg.tcl
set overwrite 0

#create output file
set msgfilename id_$msgData(message_id)_grp_$msgData(group)
while {[catch {socket -server unknown -myaddr 127.0.0.1 451} locksock]} {after 100}
set outf [open $tmpdir/msg_$msgfilename.tmp "w"]

#create an area which is only run at startup
puts $outf "set ondata \"\""
puts $outf "if {\[file exists $tmpdir/startup_$msgfilename.flag]} then {"
puts $outf "  catch {file delete $tmpdir/startup_$msgfilename.flag}"

#backlight handling (extend timeout if this timeout is longer)
if {$msgData(to_backlight)} then {
  puts $outf "  if {\$to_backlight < $msgData(to_backlight)} then {set to_backlight $msgData(to_backlight)}"
}

#start alarm
if {$msgData(to_alarm)} then {
  puts $outf "  set ondata {ALARM=true}"
  puts $outf "  if {\$to_alarm < $msgData(to_alarm)} then {set to_alarm $msgData(to_alarm)}"
}

#start service
if {$msgData(to_service)} then {
  puts $outf "  set ondata \"\$ondata SERVICE=1\""
  puts $outf "  if {\$to_service < $msgData(to_service)} then {set to_service $msgData(to_service)}"
}

#start service blink
if {$msgData(to_sblink)} then {
  puts $outf "  set ondata \"\$ondata INSTALL_MODE=true\""
  puts $outf "  if {\$to_sblink < $msgData(to_sblink)} then {set to_sblink $msgData(to_sblink)}"
}

#send UDP to set states
if {$msgData(to_service) || $msgData(to_sblink) || $msgData(to_alarm)} then {
  puts $outf "  if {\[string length \$ondata]} then {"
  puts $outf "    set udpdata \[join \"\[eval list REGAHSS \$ondata] {}\" {\\n}]" 
  puts $outf "    catch {exec echo -e \$udpdata | /etc/config/addons/lcd_msg/socat -u stdin UDP4:127.0.0.1:8182}"
  puts $outf "  }"
}

#start backlight and flash
if {$msgData(flash) || $msgData(to_backlight)} then {
  #convert timeout back to duration (value given might be a fixed time)
  puts $outf "  set backlight_duration \[expr \"$msgData(to_backlight) - \[clock seconds]\"]"
  puts $outf "  exec /sbin/daemonize /bin/tclsh /www/addons/lcd_msg/flash.tcl $msgData(flash) \$backlight_duration"
}

#of startup area
puts $outf "\} "
  
#handle message-timeout
set msgTimeout $msgData(to_message)
if { $msgTimeout } then {
  puts $outf "if {\[clock seconds] >= $msgTimeout || \$deletemsg} then \{"
} else {
  puts $outf "if {\$deletemsg} then \{"
}
puts $outf "  set skipfile 1"
puts $outf "  catch \"exec $msgData(on_delete) $msgData(message_id) $msgData(group) [clock seconds] \$overwrite\""
puts $outf "  eval file delete \[glob -nocomplain $tmpdir/*_$msgfilename.*]"
puts $outf "  set offdata \"\""
if {$msgData(to_alarm)} then {
  puts $outf "  if {$msgData(to_alarm) >= \[clock seconds]} then {set to_alarm 0; set offdata \"ALARM=false\"}"
}
if {$msgData(to_service)} then {
  puts $outf "  if {$msgData(to_service) >= \[clock seconds]} then {set to_service 0; set offdata \"\$offdata SERVICE=0\"}"
}
if {$msgData(to_sblink)} then {
  puts $outf "  if {$msgData(to_sblink) >= \[clock seconds]} then {set to_sblink 0; set offdata \"\$offdata INSTALL_MODE=false\"}"
}
if {$msgData(to_alarm) || $msgData(to_service) || $msgData(to_sblink)} then {
  puts $outf "  set udpdata \[join \"\[eval list REGAHSS \$offdata] {}\" \\\\n]"
  puts $outf "  if {\[string length \$offdata]} then {catch {exec echo -e \$udpdata | /etc/config/addons/lcd_msg/socat -u stdin UDP4:127.0.0.1:8182}}"
}

#use catch here because variables do not exist if called from delete_msg
puts $outf "  catch {set lmv_files \[lreplace \$lmv_files \$lmv_currentfileindex \$lmv_currentfileindex]}"
puts $outf "}"
puts $outf "if { \$skipfile == 0 } then \{"
puts $outf "  set readfile 0"


#write menu
set counter 1
set msglines [split $msglines "|"]
foreach msgline $msglines {
  if { [llength $msglines] == 1 } then {
    #single-page message, display only this message

    if {$msgData(rolling)} then {
      #convert spaces
      while {[regexp {(.*)?<skip +count="?(\d+)"?/>(.*)} $msgline dummy start spaces end]} {
        set msgline "$start[string repeat " " $spaces]$end"
      }
      while {[regexp {(.*)?<skip/>(.*)} $msgline dummy start end]} {
        set msgline "$start $end"
      }
      
      #split line
      if {![regexp {(^.*)?<br/>(.*)} $msgline dummy msgline1 msgline2]} then {
        set msgline1 $msgline
        set msgline2 ""
      }
      set chunklength1 [string length $msgline1]
      set chunklength2 [string length $msgline2]
      set textchunks1 [expr {$chunklength1-10}]
      set textchunks2 [expr {$chunklength2-10}]
      if {$textchunks1 < 1} then {set textchunks1 1}
      if {$textchunks2 < 1} then {set textchunks2 1}
      set maxchunks $textchunks1
      if {$textchunks1 < $textchunks2} then {set maxchunks $textchunks2}
    } else {
      set maxchunks 1
    }
    
    #generate text div (multiple for rolling text)
    for {set i 0} {$i < $maxchunks} {incr i} {
      if {$msgData(rolling)} then {
        if {$i < $textchunks1} then {
          set chunk1 [string range $msgline1 $i [expr {$i+10}]]
          if {[regexp {^( +)(.*)} $chunk1 dummy spaces end]} then {set chunk1 "<skip count=[string length $spaces]/>$end"}
        }
        if {$i < $textchunks2} then {
          set chunk2 [string range $msgline2 $i [expr {$i+10}]]
          if {[regexp {^( +)(.*)} $chunk2 dummy spaces end]} then {set chunk2 "<skip count=[string length $spaces]/>$end"}
        }
        puts $outf "  puts -nonewline \"<div name=\\\"P_[expr $i+1]\\\">$chunk1<br/>$chunk2\""
      } else {
        #no rolling, just output the data
        puts $outf "  puts -nonewline \"<div name=\\\"P_1\\\">$msgline\""
      }
      puts $outf "  puts -nonewline \"<a href=\\\"#M_DEL\\\" key=2/>\""
      puts $outf "  puts -nonewline \"\[subst \$singlepage_minus_key] \[subst \$singlepage_plus_key]\""
      if {$i+1 < $maxchunks} then {
        #more text chunks to come
        if {$i == 0} then {
          puts $outf "  puts \"<a href=\\\"#P_[expr $i+2]\\\" key=7 autoswitch=[expr 1000+$msgData(rolling)]/></div>\""
        } else {
          puts $outf "  puts \"<a href=\\\"#P_[expr $i+2]\\\" key=7 autoswitch=$msgData(rolling)/></div>\""
        }
      } else {
        #last text chunk reached
        if { $msgTimeout } then {
          puts $outf "  if { \[llength \$lmv_files] > 1 || [expr {$msgTimeout-6}] >= \[clock seconds] } then {"
          puts $outf "    puts \"<a \[subst \$next_msg] key=7 autoswitch=2000/></div>\""
          puts $outf "  } else {"
          puts $outf "    puts \"<a \[subst \$next_msg] key=7 autoswitch=\[expr {$msgTimeout-\[clock seconds] }]000/></div>\""
          puts $outf "  }"
        } else {
          if {$msgData(rolling)} then {
            puts $outf "  puts \"<a \[subst \$next_msg] key=7 autoswitch=[expr 1000+$msgData(rolling)]/></div>\""
          } else {
            puts $outf "  puts \"<a \[subst \$next_msg] key=7 autoswitch=2000/></div>\""
          }
        }
      }
    }
  } else {
    #multi-page message, display rotating
    puts $outf "  puts -nonewline \"<div name=\\\"P_$counter\\\">$msgline\""
    puts $outf "  puts -nonewline \"<a href=\\\"#M_DEL\\\" key=2/>\""
    if { $counter == 1 } then {
      #first page of multi-page message
      set prev_page "\[subst \$prev_msg]"
      set next_page "href=\\\"#P_[expr {$counter + 1}]\\\""
    } elseif { [llength $msglines] == $counter } then {
      #last page of multi-page message
      set prev_page "href=\\\"#P_1\\\""
      set next_page "\[subst \$next_msg]"
    } else {
      set prev_page "href=\\\"#P_1\\\""
      set next_page "href=\\\"#P_[expr {$counter + 1}]\\\""
    }
    puts $outf "  puts \"<a $prev_page key=1/><a $next_page key=4/><a $next_page key=7 autoswitch=2000/></div>\""
  }
  incr counter
}

#delete menu
puts $outf "  puts -nonewline \"<div name=\\\"M_DEL\\\">del $msgData(group)?<br/>No Menu Yes<a href=\\\"#P_1\\\" key=1/><a href=\\\"#M_01\\\" key=2/><a href=\\\"http://127.0.0.1/addons/lcd_msg/hss_index.cgi?to_alarm=\$to_alarm&to_service=\$to_service&to_sblink=\$to_sblink&to_backlight=\$to_backlight&fileindex=\$lmv_currentfileindex&deletegroup=$msgData(group)\\\" key=4/></div>\""

#of skipfile
puts $outf "\}"

close $outf

#compiled successfully, so rename file to .tcl and create startup flag
close [open $tmpdir/startup_$msgfilename.flag "w"]
file rename $tmpdir/msg_$msgfilename.tmp $tmpdir/msg_$msgfilename.tcl

close $locksock
